﻿using System;
using System.Collections.Generic;
using System.Text;

namespace SSToolBox.Windows.Forms
{
	#region Public Enum...

	public enum InputTypes
	{
		Text,
		Numeric,
		Currency,
		Email,
		NoInjectionSql
	}

	public enum InputLanguages
	{
		Farsi,
		English
	}

	public enum ActiveEmptyText
	{
		Happened,
		None
	}

	#endregion

	public class SSTextBox : System.Windows.Forms.TextBox
	{
		#region Fields ...

		private InputLanguages _inputLanguage;
		private System.Drawing.Font _farsiFont;
		private System.Drawing.Font _englishFont;
		private System.Drawing.Color _currentForeColor;
		private System.Drawing.Color _colorEmptyTextHappend;
		private System.Drawing.Color _colorMouseMoveTextbox;
		private System.Globalization.CultureInfo _farsiCulture;
		private System.Globalization.CultureInfo _englishCulture;
		private System.Windows.Forms.ErrorProvider _errorProvider;
		private bool _activeEmptyText;
		private string _caption;
		private string _textEmptyHappend;
		private string _noString;
		private ActiveEmptyText EmptyTextHappened;


		#endregion

		#region Properties Extera Features ...

		[System.ComponentModel.DefaultValue("")]
		[System.ComponentModel.Category("Extra Features")]
		public string Caption
		{
			get { return _caption; }
			set
			{
				_caption = value;
				TextEmptyHappend = value;
			}
		}

		[System.ComponentModel.DefaultValue(false)]
		[System.ComponentModel.Category("Extra Features")]
		public bool IsRequired { get; set; }

		[System.ComponentModel.DefaultValue(InputTypes.Text)]
		[System.ComponentModel.Category("Extra Features")]
		public InputTypes InputType { get; set; }

		[System.ComponentModel.DefaultValue(InputLanguages.Farsi)]
		[System.ComponentModel.Category("Extra Features")]
		public InputLanguages InputLanguage
		{
			get { return _inputLanguage; }
			set
			{
				_inputLanguage = value;

				switch (value)
				{
					case InputLanguages.Farsi:
					{
						Font = _farsiFont;
						RightToLeft = System.Windows.Forms.RightToLeft.Yes;

						break;
					}

					case InputLanguages.English:
					{
						Font = _englishFont;
						RightToLeft = System.Windows.Forms.RightToLeft.No;

						break;
					}
					default:
					break;
				}
			}
		}


		// [System.ComponentModel.DefaultValue(System.Drawing.Color.DarkGray)]
		[System.ComponentModel.Category("Extra Features")]
		public System.Drawing.Color ColorEmptyTextHappend
		{
			get { return _colorEmptyTextHappend; }
			set { _colorEmptyTextHappend = value; }
		}

		[System.ComponentModel.DefaultValue("")]
		[System.ComponentModel.Category("Extra Features")]
		private string TextEmptyHappend
		{
			get { return _textEmptyHappend; }
			set { _textEmptyHappend = value; }
		}

		[System.ComponentModel.DefaultValue(false)]
		[System.ComponentModel.Category("Extra Features")]
		public bool ActiveHiddenText
		{
			get { return _activeEmptyText; }
			set { _activeEmptyText = value; }
		}
		[System.ComponentModel.Description("No String character in textbox \n Example  /*\\@#$ ")]
		[System.ComponentModel.Category("Extra Features")]
		public string NoString
		{
			get { return _noString; }
			set { _noString = value; }
		}


		[System.ComponentModel.Category("Extra Features")]
		public System.Drawing.Color ColorMouseMoveTextbox
		{
			get { return _colorMouseMoveTextbox; }
			set { _colorMouseMoveTextbox = value; }
		}

		#endregion

		#region Properties override visible ...

		public override string Text
		{
			get
			{
				if (EmptyTextHappened == ActiveEmptyText.Happened)
					return "";
				return base.Text;
			}
			set
			{
				if (EmptyTextHappened == ActiveEmptyText.None)
				{
					//if (CheckInputChar(value) == false)
					//{
					//	base.Text = value;
					//}
					base.Text = value;
				}
				else
				{
					base.Text = value;
				}

			}
		}

		[System.ComponentModel.Browsable(false)]
		public override System.Drawing.Font Font
		{
			get
			{
				return base.Font;
			}
			set
			{
				base.Font = value;
			}
		}

		[System.ComponentModel.Browsable(false)]
		public override System.Windows.Forms.RightToLeft RightToLeft
		{
			get
			{
				return base.RightToLeft;
			}
			set
			{
				base.RightToLeft = value;
			}
		}

		#endregion

		#region Constructors ...

		public SSTextBox()
		{
			_farsiFont = new System.Drawing.Font("Tahoma", 8, System.Drawing.FontStyle.Regular);
			_englishFont = new System.Drawing.Font("Tahoma", 8, System.Drawing.FontStyle.Regular);

			_farsiCulture = new System.Globalization.CultureInfo("fa-IR");
			_englishCulture = new System.Globalization.CultureInfo("en-US");

			Caption = "";
			IsRequired = false;
			InputType = InputTypes.Text;
			InputLanguage = InputLanguages.Farsi;
			EmptyTextHappened = ActiveEmptyText.None;
			ColorEmptyTextHappend = System.Drawing.Color.DarkGray;
			ColorMouseMoveTextbox = System.Drawing.Color.DarkGray;
			CheckEmptyHappend(true);
		}


		#endregion

		#region Methods ...

		public void ShowHiddenText()
		{
			CheckEmptyHappend(true);
		}

		private void CheckEmptyHappend(bool Happend)
		{
			if (ActiveHiddenText)
			{
				if ((Text.Trim() == string.Empty && EmptyTextHappened == ActiveEmptyText.None) || Happend)
				{
					EmptyTextHappened = ActiveEmptyText.Happened;
					Font = new System.Drawing.Font(InputLanguage == InputLanguages.English ? _englishFont.FontFamily : _farsiFont.FontFamily, 8, System.Drawing.FontStyle.Italic);
					Text = TextEmptyHappend;

					if (_currentForeColor == System.Drawing.Color.Empty)
						_currentForeColor = ForeColor;

					ForeColor = ColorEmptyTextHappend;
				}
				else
				{
					EmptyTextHappened = ActiveEmptyText.None;
					Font = new System.Drawing.Font(InputLanguage == InputLanguages.English ? _englishFont.FontFamily : _farsiFont.FontFamily, 8, System.Drawing.FontStyle.Regular);
					ForeColor = _currentForeColor;
					_currentForeColor = System.Drawing.Color.Empty;
					Text = string.Empty;
				}
			}
		}

		/// <summary>
		/// barae vared nakardane charakter hae khas mibashad
		/// </summary>
		/// <param name="ch"></param>
		/// <returns></returns>
		private bool CheckNoString(char ch)
		{
			if (!string.IsNullOrEmpty(NoString))
			{
				foreach (char item in NoString)
				{
					if (item == ch)
						return true;
				}
			}
			return false;
		}

		private bool CheckInputChar(string strInput)
		{
			for (int index = 0; index < strInput.Length; index++)
			{
				if (CheckInputChar(strInput[index]))
				{
					return true;
				}
			}
			return false;
		}

		private bool CheckInputChar(char chInput)
		{
			bool result = false;

			switch (InputType)
			{
				case InputTypes.Numeric:
				{
					if (!(char.IsDigit(chInput) || chInput == 8))
					{
						result = true;
						InputReleaseError(this, new EventReleaseErrorArgs { EnumMessage = EunmException.InputNoNumeric });
					}
					break;
				}
				case InputTypes.Currency:
				{
					if (!(char.IsDigit(chInput) || chInput == 8))
					{
						result = true;
						InputReleaseError(this, new EventReleaseErrorArgs { EnumMessage = EunmException.InputNoCurrency });
					}
					break;
				}
				case InputTypes.Email:
				{
					if (!(char.IsLetterOrDigit(chInput) || chInput == '@' || chInput == '.' || chInput == '_' || chInput == '-' || chInput == 8))
					{
						result = true;
						InputReleaseError(this, new EventReleaseErrorArgs { EnumMessage = EunmException.InputNoEmail });
					}
					break;
				}
				case InputTypes.NoInjectionSql:
				{
					result = PublicClass.CheckInjectionSql(chInput);
					if (result)
					{
						InputReleaseError(this, new EventReleaseErrorArgs { EnumMessage = EunmException.InputInjectionSql });
					}
					break;
				}
				default:

				break;
			}
			if (result == false)
			{
				result = CheckNoString(chInput);
				if (result)
				{
					InputReleaseError(this, new EventReleaseErrorArgs { EnumMessage = EunmException.InputStringError });
				}
			}

			return (result);
		}
		#endregion

		#region Events Override...

		protected override void OnTextChanged(EventArgs e)
		{
			base.OnTextChanged(e);

			if (PublicClass.CheckInjectionSql(this.Text))
			{
				this.Text = string.Empty;
			}
		}

		protected override void OnKeyPress(System.Windows.Forms.KeyPressEventArgs e)
		{
			base.OnKeyPress(e);

			e.Handled = CheckInputChar(e.KeyChar);
		}

		protected override void OnValidating(System.ComponentModel.CancelEventArgs e)
		{
			base.OnValidating(e);

			if (IsRequired)
			{
				if (_errorProvider == null)
					_errorProvider = new System.Windows.Forms.ErrorProvider();

				if (Text.Trim().Length == 0 || EmptyTextHappened == ActiveEmptyText.Happened)
				{
					CheckEmptyHappend(true);
					switch (InputLanguage)
					{
						case InputLanguages.Farsi:
						{
							_errorProvider.SetIconAlignment(this, System.Windows.Forms.ErrorIconAlignment.MiddleLeft);
							_errorProvider.SetError(this, string.Format("شما فیلد مربوط به {0} را پر نکرده اید", Caption));

							break;
						}

						case InputLanguages.English:
						{
							_errorProvider.SetIconAlignment(this, System.Windows.Forms.ErrorIconAlignment.MiddleRight);
							_errorProvider.SetError(this, string.Format("{0} field should not be empty", Caption));

							break;
						}
						default:
						break;
					}

					// e.Cancel = true;
				}
				else
				{
					_errorProvider.SetError(this, "");
					EmptyTextHappened = ActiveEmptyText.None;
				}
			}
			else if (Text.Trim().Length == 0)
			{
				CheckEmptyHappend(true);
			}

		}

		protected override void OnValidated(EventArgs e)
		{
			base.OnValidated(e);

			if (Text.Trim().Length != 0 && EmptyTextHappened == ActiveEmptyText.None)
			{
				switch (InputType)
				{
					case InputTypes.Currency:
					{
						long lngData = long.Parse(Text.Trim());

						switch (InputLanguage)
						{
							case InputLanguages.Farsi:
							{
								Text = lngData.ToString("#,##0 ریال");

								break;
							}

							case InputLanguages.English:
							{
								Text = lngData.ToString("#,##0 $");

								break;
							}

							default:
							break;
						}

						break;
					}
					default:
					break;
				}
			}
		}

		protected override void OnEnter(EventArgs e)
		{
			base.OnEnter(e);

			if (EmptyTextHappened == ActiveEmptyText.Happened)
				CheckEmptyHappend(false);

			switch (InputType)
			{
				case InputTypes.Currency:
				{
					Text = Text.Replace("ریال", "").Replace("$", "").Replace(",", "").Replace(" ", "");

					break;
				}

				default:
				break;
			}

			if (Multiline)
			{
				SelectionStart = 0;
				DeselectAll();
			}
			else
				SelectAll();


			switch (InputLanguage)
			{
				case InputLanguages.Farsi:
				{
					System.Windows.Forms.InputLanguage.CurrentInputLanguage = System.Windows.Forms.InputLanguage.FromCulture(_farsiCulture);

					break;
				}

				case InputLanguages.English:
				{
					System.Windows.Forms.InputLanguage.CurrentInputLanguage = System.Windows.Forms.InputLanguage.FromCulture(_englishCulture);

					break;
				}
				default:
				break;
			}
		}

		protected override void OnMouseMove(System.Windows.Forms.MouseEventArgs e)
		{
			base.OnMouseMove(e);
			if (EmptyTextHappened == ActiveEmptyText.Happened)
				ForeColor = ColorMouseMoveTextbox;
		}

		protected override void OnMouseLeave(EventArgs e)
		{
			base.OnMouseLeave(e);
			if (EmptyTextHappened == ActiveEmptyText.Happened)
				ForeColor = ColorEmptyTextHappend;
		}

		#endregion

		#region Event...

		public event ReleaseErrorDelegate OnInputReleaseError;

		private void InputReleaseError(object sender, EventReleaseErrorArgs e)
		{
			if (OnInputReleaseError != null)
			{
				OnInputReleaseError(sender, e);
			}
		}

		#endregion

	}
}
