<?php
  // ----------------------------------------------------------------------
  // LICENSE
  
  // This program is open source product; you can redistribute it and/or
  // modify it under the terms of the GNU Lesser General Public License (LGPL)
  // as published by the Free Software Foundation; either version 3
  // of the License, or (at your option) any later version.
  
  // This program is distributed in the hope that it will be useful,
  // but WITHOUT ANY WARRANTY; without even the implied warranty of
  // MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  // GNU Lesser General Public License for more details.
  
  // You should have received a copy of the GNU Lesser General Public License
  // along with this program.  If not, see <http://www.gnu.org/licenses/>.
  // ----------------------------------------------------------------------
  // Class Name: FaGlyphs (Persian Glyphs) is a simple class to render Persian text
  // Filename:   FaGlyphs.class.php
  // Author:     Shamim Rezaie, <http://rezaie.info>
  // Original    Author(s): Khaled Al-Sham'aa <khaled@ar-php.org>
  // Purpose:    This class takes Persian text (encoded in Windows-1256 character 
  //             set) as input and performs Persian glyph joining on it and outputs 
  //             a UTF-8 hexadecimals stream that is no longer logically arranged 
  //             but in a visual order which gives readable results when formatted 
  //             with a simple Unicode rendering just like GD and UFPDF libraries 
  //             that does not handle basic connecting glyphs of Persian language 
  //             yet but simply outputs all stand alone glyphs in left-to-right 
  //             order.
  // ----------------------------------------------------------------------
  // This class is a modified version of ArGlyphs class which has been developed
  // by Khaled Al-Sham'aa <khaled@ar-php.org>.
  // It has been modified to be compatible with Persian texts and also to be able
  // to handle more characters such as ZWNJ, ZWJ, , , , and .


/**
 * Persian Glyphs is a simple class to render Persian text
 *
 * PHP class to render Persian text by performs Persian glyph joining on it,
 * then output a UTF-8 hexadecimals stream gives readable results on PHP
 * libraries supports UTF-8.
 *
 * @author Shamim Rezaie <http://rezaie.info>
 */
class FaGlyphs
{
    private $glyphs = null;
    private $hex = null;
    private $prevLink;
    private $nextLink;

    private $crntChar = null;
    private $prevChar = null;
    private $nextChar = null;

    public function __construct()
    {
        $this->prevLink = chr(0x98).chr(0x9E).'';
        $this->nextLink = chr(0x98).chr(0x9E).''.chr(0x8E).'';
        $this->vowel = '';

// isolated    final    initial    medial
        $this->glyphs = '';
        $this->hex = '064B064B064B064B064C064C064C064C064D064D064D064D064E064E064E064E064F064F064F064F065006500650065006510651065106510652065206520652';

// p ch zh g k
        $this->glyphs .= ''.chr(0x8E).''.chr(0x98);
        $this->hex .= 'FB56FB57FB58FB59'.'FB7AFB7BFB7CFB7D'.'FB8AFB8BFB8AFB8B'.'FB92FB93FB94FB95'.'FB8EFB8FFB90FB91';

// ZWNJ ZWJ
        $this->glyphs .= chr(0x9D).chr(0x9E);
        $this->hex .= '200C200C200C200C'.'200D200D200D200D';

        $this->glyphs .= '׫';
        $this->hex .= '00D700D700D700D7'.'00BB00BB00BB00BB'.'00AB00AB00AB00AB'.'00F700F700F700F7';

        $this->glyphs .= '';
        $this->hex .= 'FE80FE80FE80FE80FE81FE82FE81FE82FE83FE84FE83FE84FE85FE86FE85FE86FE87FE88FE87FE88FE89FE8AFE8BFE8CFE8DFE8EFE8DFE8EFE8FFE90FE91FE92';

        $this->glyphs .= '';
        $this->hex .= 'FE93FE94FE93FE94FE95FE96FE97FE98FE99FE9AFE9BFE9CFE9DFE9EFE9FFEA0FEA1FEA2FEA3FEA4FEA5FEA6FEA7FEA8FEA9FEAAFEA9FEAAFEABFEACFEABFEAC';

        $this->glyphs .= '';
        $this->hex .= 'FEADFEAEFEADFEAEFEAFFEB0FEAFFEB0FEB1FEB2FEB3FEB4FEB5FEB6FEB7FEB8FEB9FEBAFEBBFEBCFEBDFEBEFEBFFEC0FEC1FEC2FEC3FEC4FEC5FEC6FEC7FEC8';

        $this->glyphs .= '';
        $this->hex .= 'FEC9FECAFECBFECC'.'FECDFECEFECFFED0'.'FED1FED2FED3FED4'.'FED5FED6FED7FED8'.'FED9FEDAFEDBFEDC'.'FEDDFEDEFEDFFEE0'.'FEE1FEE2FEE3FEE4'.'FEE5FEE6FEE7FEE8';

        $this->glyphs .= 'ܡ';
        $this->hex .= 'FEE9FEEAFEEBFEEC'.'FEEDFEEEFEEDFEEE'.'FEEFFEF0FEEFFEF0'.'FBFCFBFDFBFEFBFF'.'0640064006400640'.'060C060C060C060C'.'061F061F061F061F'.'061B061B061B061B';
    }

    private function get_glyphs($char, $type)
    {
        $pos = strpos($this->glyphs, $char);

        $pos = $pos*16 + $type*4;

        return substr($this->hex, $pos, 4);
    }

    /**
     * @return String Persian glyph joining in UTF-8 hexadecimals stream
     * @param String $str Persian string in Windows-1256 charset
     * @desc Convert Persian Windows-1256 charset string into glyph
     *               joining in UTF-8 hexadecimals stream
     * @author Khaled Al-Shamaa <khaled.alshamaa@gmail.com>
     */
    private function pre_convert($str)
    {
        $crntChar = null;
        $prevChar = null;
        $nextChar = null;
        $output = '';

        $chars = preg_split('//', $str);
        $max = count($chars);

        for ($i = $max - 1; $i >= 0; $i--) {
            $crntChar = $chars[$i];
            if ($i > 0){
                $prevChar = $chars[$i - 1];
            }else{
                $prevChar = NULL;
            }

            if ($prevChar && strpos($this->vowel, $prevChar) !== false) {
                $prevChar = $chars[$i - 2];
                if ($prevChar && strpos($this->vowel, $prevChar) !== false) {
                    $prevChar = $chars[$i - 3];
                }
            }

            $Reversed = false;
            $flip_arr = ')]>}';
            $ReversedChr = '([<{';

            if ($crntChar && strpos($flip_arr, $crntChar) !== false) {
                $crntChar = substr($ReversedChr, strpos($flip_arr, $crntChar), 1);
                $Reversed = true;
            } else {
                $Reversed = false;
            }

            if ($crntChar && (strpos($ReversedChr, $crntChar) !== false) && !$Reversed) {
                $crntChar = substr($flip_arr, strpos($ReversedChr, $crntChar), 1);
            }

            if ($crntChar && strpos($this->vowel, $crntChar) !== false) {
                if ((strpos($this->nextLink, $chars[$i + 1]) !== false)  && (strpos($this->prevLink, $prevChar) !== false)) {
                    $output .= '&#x' . $this->get_glyphs($crntChar, 1) . ';';
                } else {
                    $output .= '&#x' . $this->get_glyphs($crntChar, 0) . ';';
                }
                continue;
            }


            if (ord($crntChar) < 128) {
                $output .= $crntChar;
                $nextChar = $crntChar;
                continue;
            }

            $form = 0;

            if ($prevChar && strpos($this->prevLink, $prevChar) !== false) {
                $form++;
            }
            if ($nextChar && strpos($this->nextLink, $nextChar) !== false) {
                $form += 2;
            }

            $output .= '&#x' . $this->get_glyphs($crntChar, $form) . ';';
            $nextChar = $crntChar;
        }

        $output = $this->decode_entities($output, $exclude = array('&'));
        return $output;
    }

    /**
     * @return String Persian glyph joining in UTF-8 hexadecimals stream (take
     *                care of whole document including English sections as well
     *                as numbers and arcs etc...)
     * @param String $str Persian string in Windows-1256 charset
     * @param Integer $max_chars Max number of chars you can fit in one line
     * @param Boolean $hindo If true use Hindo digits else use Arabic digits
     * @desc Convert Persian Windows-1256 charset string into glyph
     *               joining in UTF-8 hexadecimals stream (take care of whole
     *               the document including English sections as well as numbers
     *               and arcs etc...)
     * @author Khaled Al-Shamaa <khaled.alshamaa@gmail.com>
     */
    public function convert($str, $max_chars = 50, $hindo = true)
    {
        $str = str_replace(array("\r\n", "\n", "\r"), "\n", $str);

        $lines = array();
        $words = split(' ', $str);
        $w_count = count($words);
        $c_chars = 0;
        $c_words = array();

        $english = array();
        $en_index = -1;

        for ($i = 0; $i < $w_count; $i++) {
            if (preg_match("/^[a-z\d\\/\!\@\#\$\%\^\&\*\(\)\-\_\+\=\~\:\"\'\[\]\{\}\;\,\.\|]*$/i", $words[$i])) {
                $words[$i] = strrev($words[$i]);
                array_push($english, $words[$i]);
                if ($en_index == -1) {
                    $en_index = $i;
                }
            } elseif ($en_index != -1) {
                $en_count = count($english);

                for ($j = 0; $j < $en_count; $j++) {
                    $words[$en_index + $j] = $english[$en_count - 1 - $j];
                }

                $en_index = -1;
                $english = array();
            }

            $en_count = count($english);

            for ($j = 0; $j < $en_count; $j++) {
                $words[$en_index + $j] = $english[$en_count - 1 - $j];
            }
        }

        for ($i = 0; $i < $w_count; $i++) {
            $w_len = strlen($words[$i]) + 1;

              
            if ($c_chars + $w_len < $max_chars) {
                if (preg_match("/\n/i", $words[$i])) {
                    $words_nl = split("\n", $words[$i]);

                    array_push($c_words, $words_nl[0]);
                    array_push($lines, implode(' ', $c_words));

                    $nl_num = count($words_nl) - 1;
                    for ($j = 1; $j < $nl_num; $j++) {
                        array_push($lines, $words_nl[$j]);
                    }

                    $c_words = array($words_nl[$nl_num]);
                    $c_chars = strlen($words_nl[$nl_num]) + 1;
                } else {
                    array_push($c_words, $words[$i]);
                    $c_chars += $w_len;
                }
            } else {
                array_push($lines, implode(' ', $c_words));
                $c_words = array($words[$i]);
                $c_chars = $w_len;
            }
        }
        array_push($lines, implode(' ', $c_words));

        $max_line = count($lines);
        $output = '';
        for ($j = $max_line - 1; $j >= 0; $j--) {
            $output .= $lines[$j] . "\n";
        }

        $output = rtrim($output);

        $output = $this->pre_convert($output);
        if ($hindo) {
            $Nums = array(0, 1, 2, 3, 4, 5, 6, 7, 8, 9);
            $arNums = array('٠', '١', '٢', '٣', '٤', '٥', '٦', '٧', '٨', '٩');
            $output = str_replace($Nums, $arNums, $output);
        }

        return $output;
    }

    /**
     * @param String $text The text to decode entities in.
     * @param Array $exclude An array of characters which should not be decoded.
     *                       For example, array('<', '&', '"'). This affects
     *                       both named and numerical entities.
     * @desc Decode all HTML entities (including numerical ones)
     *              to regular UTF-8 bytes. Double-escaped entities will only be
     *              decoded once ("&amp;lt;" becomes "&lt;", not "<").
     */
    private function decode_entities($text, $exclude = array())
    {
        static $table;

        // We store named entities in a table for quick processing.
        if (!isset($table)) {
            // Get all named HTML entities.
            $table = array_flip(get_html_translation_table(HTML_ENTITIES));

            // PHP gives us ISO-8859-1 data, we need UTF-8.
            $table = array_map('utf8_encode', $table);

            // Add apostrophe (XML)
            $table['&apos;'] = "'";
        }
        $newtable = array_diff($table, $exclude);

        // Use a regexp to select all entities in one pass, to avoid decoding 
        // double-escaped entities twice.
        return preg_replace('/&(#x?)?([A-Za-z0-9]+);/e', '$this
          ->_decode_entities("$1", "$2", "$0", $newtable, $exclude)', $text);
    }

    /**
     * Helper function for decode_entities
     */
    private function _decode_entities($prefix, $codepoint, $original, &$table, &$exclude)
    {
        // Named entity
        if (!$prefix) {
            if (isset($table[$original])) {
                return $table[$original];
            } else {
                return $original;
            }
        }

        // Hexadecimal numerical entity
        if ($prefix == '#x') {
            $codepoint = base_convert($codepoint, 16, 10);
        }

        // Encode codepoint as UTF-8 bytes
        if ($codepoint < 0x80) {
            $str = chr($codepoint);
        } elseif ($codepoint < 0x800) {
            $str = chr(0xC0 | ($codepoint >> 6)) . chr(0x80 | ($codepoint & 0x3F));
        } elseif ($codepoint < 0x10000) {
            $str = chr(0xE0 | ($codepoint >> 12)) . chr(0x80 | (($codepoint >> 6) & 0x3F)) . chr(0x80 | ($codepoint & 0x3F));
        } elseif ($codepoint < 0x200000) {
            $str = chr(0xF0 | ($codepoint >> 18)) . chr(0x80 | (($codepoint >> 12) & 0x3F)) . chr(0x80 | (($codepoint >> 6) & 0x3F)) . chr(0x80 | ($codepoint & 0x3F));
        }

        // Check for excluded characters
        if (in_array($str, $exclude)) {
            return $original;
        } else {
            return $str;
        }
    }
}

?>
